﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.Controls.Menus.js
//
/// <reference path="..\DevTime.js" />
/// <reference path="..\jquery-1.4.3.js" />
/// <reference path="..\MicrosoftAjax.debug.js"/>
/// <reference path="TFS.debug.js" />
/// <reference path="TFS.Core.debug.js" />
/// <reference path="TFS.Diag.debug.js" />
/// <reference path="TFS.UI.debug.js" />
/// <reference path="TFS.UI.Controls.js" />

TFS.module("TFS.UI.Controls.Menus", ["TFS.Resources.Common", "TFS.UI.Controls", "TFS.Core", "TFS.UI", "TFS.Diag"], function () {
    var log = TFS.Diag.log,
        assert = TFS.Diag.assert,
        verbose = TFS.Diag.LogVerbosity.Verbose,
        delegate = TFS.Core.delegate,
        domElem = TFS.UI.domElem,
        Diag = TFS.Diag,
        Core = TFS.Core,
        KeyCode = TFS.UI.KeyCode,
        handleError = TFS.handleError,
        getErrorMessage = TFS.getErrorMessage,
		Core = TFS.Core,
        Controls = TFS.UI.Controls,
        Resources = TFS.Resources.Common,
		MenuItemState,
        MenuType,
        MenuAlign,
        MenuOrientation,
        menuManager;

    MenuOrientation = {
        Horizontal: 0,
        Vertical: 1
    };

    MenuOrientation.parse = function (type) {
        if (type === "vertical") {
            return MenuOrientation.Vertical;
        }
        return MenuOrientation.Horizontal;
    };

    MenuType = {
        None: 0,
        Normal: 1,
        Static: 2,
        Popup: 3,
        DropDown: 4,
        Chevron: 5,
        SubMenu: 6
    };

    MenuItemState = {
        None: 0,
        Disabled: 0x01,
        Hidden: 0x02
    };

    function toMenuItemState(command) {
        var state;

        if (command) {
            state = command.state;

            if (command.disabled) {
                state = state | MenuItemState.Disabled;
            }

            if (command.hidden) {
                state = state | MenuItemState.Hidden;
            }
        }

        return state;
    }

    MenuType.parse = function (type) {
        switch (type) {
            case "static":
                return MenuType.Static;
            case "popup":
                return MenuType.Popup;
            case "dropdown":
                return MenuType.DropDown;
            case "chevron":
                return MenuType.Chevron;
        }
        return MenuType.Normal;
    };

    MenuAlign = {
        RightBottom: 0,
        RightJustify: 1,
        LeftBottom: 2
    };

    MenuAlign.parse = function (type) {
        switch (type) {
            case "right-bottom":
                return MenuAlign.RightBottom;
            case "left-bottom":
                return MenuAlign.LeftBottom;
        }
        return MenuAlign.RightJustify;
    };


    menuManager = (function () {
        /// <summary>Menu manager keeps track of command states of all type of menu types</summary>

        function MenuManager() {
            this._commandStates = {};
        }

        MenuManager.inheritBehavior(Core.EventListBehavior);

        $.extend(MenuManager.prototype, {
            _executeEvent: "execute-command",
            _commandStates: null,

            getCommandState: function (commandId, context) {
                var state = this._commandStates && this._commandStates[commandId];

                if ($.isFunction(state)) {
                    return state.call(context, commandId, context);
                }

                return state;
            },

            updateCommandStates: function (commands) {
                var that = this;

                if (commands) {
                    $.each(commands, function (i, command) {
                        that._commandStates[command.id] = toMenuItemState(command);
                    });
                }
            },

            executeCommand: function (args) {
                /// <summary>Execute the command specified in the arguments.</summary>
                /// <param name="args">The arguments to pass through to the command</parma>

                this._fireEvent(this._executeEvent, this, args);
            },

            attachExecuteCommand: function (handler) {
                /// <summary>Attaches handler for the commands executed</summary>
                /// <param name="handler" type="Function">Handler to attach to be executed when any command executed</param>
                this.attachEvent(this._executeEvent, handler);
            },

            detachExecuteCommand: function (handler) {
                /// <summary>Detaches handler for the commands executed</summary>
                /// <param name="handler" type="Function">Handler to detach</param>
                this.detachEvent(this._executeEvent, handler);
            }
        });

        // MenuManager is a singleton instance. Thus, an instance of it will be exposed outside.
        return new MenuManager();
    })();

    function MenuBase(options) {
        this.baseConstructor.call(this, $.extend({}, options));

        this._type = MenuType.parse(this._options.type);
        this._children = [];
    }

    MenuBase.inherit(Controls.BaseControl, {
        _type: null,
        _parent: null,
        _children: null,
        _commandStates: null,

        actionArguments: null,

        getOwner: function () {
            return this._parent ? this._parent.getOwner() : this;
        },

        getContextInfo: function () {
            return this._options.contextInfo;
        },

        getActionArguments: function () {
            var args = this._options.arguments || this.actionArguments;

            if (typeof args === "string") {
                // String argument means a URL. We are creating
                // an action object with url as a parameter

                return {
                    url: args
                };
            }
            else if ($.isFunction(args)) {
                return args.call(this, this.getOwner().getContextInfo());
            }

            return args;
        },

        getCommandState: function (commandId, context) {
            var state = this._commandStates && this._commandStates[commandId];

            context = context || this;

            //intented == operator
            if (state == null && $.isFunction(this._options.getCommandState)) {
                state = this._options.getCommandState.call(context, commandId, context);
            }

            //intented == operator
            if (state == null) {
                if (this._parent) {
                    return this._parent.getCommandState(commandId, context);
                }
                else {
                    return menuManager.getCommandState(commandId, context);
                }
            }

            if ($.isFunction(state)) {
                return state.call(context, commandId, context);
            }

            return state;
        },

        _updateCommandStates: function (commands) {
            var that = this;
            if (!this._commandStates) {
                this._commandStates = {};
            }

            if (commands) {
                $.each(commands, function (i, command) {
                    that._commandStates[command.id] = toMenuItemState(command);
                });
            }
        },

        updateCommandStates: function (commands) {
            this._updateCommandStates(commands);
        },

        _fireUpdateCommandStates: function (context) {
            context = context || this;

            if ($.isFunction(this._options.updateCommandStates)) {
                if (this._options.updateCommandStates.call(context, context) === false) {
                    return;
                }
            }

            if (this._parent) {
                this._parent._fireUpdateCommandStates(context);
            }
        },

        _clear: function () {
            var i, len = this._children.length;

            for (i = 0; i < len; i++) {
                this._children[i]._clear();
                this._children[i].dispose();
            }

            this._children = [];
        }
    });

    function MenuItem(options) {
        this.baseConstructor.call(this, $.extend({ cssCoreClass: "menu-item", tagName: "li" }, options));

        this._item = this._options.item || {};
        this.actionArguments = this._item.arguments;
        this._align = MenuAlign.parse(this._options.align);
    }

    MenuItem.extend({
        _typeName: "tfs.menu.item",
        getScopedCommandId: function (id, scope) {
            return scope ? scope + "-" + id : id;
        }
    });

    MenuItem.inherit(MenuBase, {
        _item: null,                // Holds information like text, icon, child items, etc.
        _align: null,               // Sub menu align
        _index: 0,                  // Index within siblings
        _highlightHover: false,     // Indicates the 'highlighted' state of the item
        _highlightPressed: false,   // Indicates the 'pressed' state of the item

        getCommandId: function () {
            return MenuItem.getScopedCommandId(this._item.id, this._item.scope);
        },
        getAction: function () {
            var idIsAction = this._item.idIsAction;
            if (idIsAction === false) {
                return this._item.action;
            }

            return this._item.action || this.getCommandId();
        },
        hasAction: function () {
            var action = this.getAction();
            return action !== null && typeof action !== "undefined";
        },
        hasSubMenu: function () {
            return !this.isDefault() && this._item.childItems;
        },
        isDecorated: function () {
            return this._item.decorated === true;
        },
        isDefault: function () {
            return this._type === MenuType.None;
        },
        isSeparator: function () {
            return this._item.separator === true;
        },
        isSelected: function () {
            return this._item.selected === true;
        },
        getCommandState: function (commandId, context) {
            var state;

            commandId = commandId || this._item.id;
            context = context || this;

            if (commandId == this._item.id) {
                state = toMenuItemState(this._item);
            }

            //intented == operator
            if (state == null) {
                return this._base(commandId, context);
            }

            if ($.isFunction(state)) {
                return +state.call(context, commandId, context);
            }

            return +state;
        },
        isHidden: function () {
            return (this.getCommandState() & MenuItemState.Hidden) == MenuItemState.Hidden;
        },
        isEnabled: function () {
            if (this.isSeparator()) {
                return false;
            }

            return (this.getCommandState() & (MenuItemState.Disabled | MenuItemState.Hidden)) == MenuItemState.None;
        },
        initialize: function () {
            this._base();

            this._decorate();
        },

        _attachMenuEvents: function () {

        },

        update: function (item) {
            this._clear();
            this._item = item;
            this._item.decorated = false;
            this._element.empty();
            this._unbind("click mouseover mouseout mousedown mouseup");
            this._decorate();
        },

        updateItems: function (items) {
            this._clear();
            this._item.childItems = items;
        },

        _createIconElement: function () {
            return $(domElem("span", "icon"));
        },

        _createTextElement: function () {
            return $(domElem("span", "text"));
        },

        _createDropElement: function () {
            return $(domElem("span", "drop"));
        },

        _createSeparatorElement: function () {
            return $(domElem("div", "separator"));
        },

        _decorate: function () {
            var element = this._element,
                hasIcon = this.getOwner()._options.showIcon === true,
                hasChildren = this.hasSubMenu(),
				title,
                hasText = this._item.showText !== false,
                separatorElement,
                iconElement,
                textElement,
                dropElement,
                containsIcon = false,
                containsText = false;

            element.attr("id", "mi_" + Controls.getId());

            if (!this.isDecorated()) {

                title = this._item.title;
                if (!title && !this._item.encoded) {
                    title = this._item.text;
                }

                // Adding core classes for menu item
                element
                    .addClass(this._options.cssCoreClass)
                    .addClass(this._options.cssClass)
                    .attr("title", title);

                // This is necessary for tests, current menu item to be identified
                if (this._item.id) {
                    element.attr("command", this._item.id);
                }

                if (this._item.cssClass) {
                    this._element.addClass(this._item.cssClass);
                }

                if (this.isSeparator()) {
                    separatorElement = this._createSeparatorElement();
                    if (this._item.text) {
                        separatorElement.text(this._item.text);
                        separatorElement.removeClass("separator");
                        separatorElement.addClass("text-separator");
                    }

                    element.append(separatorElement);
                }
                else {
                    if (hasIcon && !this._item.noIcon) {
                        iconElement = this._createIconElement();

                        if ($.isFunction(this._item.icon)) {
                            iconElement.append(this._item.icon.call(this, iconElement));
                        }
                        else if (this._item.icon) {
                            iconElement.addClass(this._item.icon);
                        }

                        element.append(iconElement);
                        containsIcon = true;
                    }

                    if (hasText) {
                        textElement = this._createTextElement();
                        if (this._item.encoded) {
                            textElement[0].innerHTML = this._item.text || "";
                        }
                        else {
                            if (typeof (this._item.text) === "string") {
                                textElement.text(this._item.text);
                            }
                            else if ($.isFunction(this._item.text)) {
                                textElement.text(this._item.text.call(this) || "");
                            }
                        }
                        containsText = true;
                        element.append(textElement);
                    }

                    if (!containsText && containsIcon) {
                        element.addClass('icon-only');
                    }

                    if (hasChildren && !this._item.hideDrop) {
                        dropElement = this._createDropElement();
                        element.append(dropElement);
                        element.addClass("drop-visible");
                    }
                }
            }
            else {
                dropElement = element.children(".drop").eq(0);
            }

            if (!this.isEnabled()) {
                // Adding disable class to the menu item
                element.addClass("disabled");
            } else {
                element.removeClass("disabled");
            }

            if (this.isHidden()) {
                // Adding disable class to the menu item
                element.addClass("invisible");
            } else {
                element.removeClass("invisible");
            }

            if (this.isSelected()) {
                // Adding selector class to the menu item
                element.addClass("selected");
            } else {
                element.removeClass("selected");
            }

            if (!this.isDefault()) {
                this._bind("click", delegate(this, this._onClick));

                if (!this.isSeparator()) {
                    this._bind("mouseover", delegate(this, this._onMouseOver));
                    this._bind("mouseout", delegate(this, this._onMouseOut));
                    this._bind("mousedown", delegate(this, this._onMouseDown)); 
                    this._bind("mouseup", delegate(this, this._onMouseUp)); 

                    if (hasChildren && dropElement && dropElement.length > 0) {
                        this._bind(dropElement, "click", delegate(this, this._onDropClick));
                    }
                }
            }

            if (this.getOwner()._options.markUnselectable === true) {
                TFS.UI.makeElementUnselectable(element[0]);
            }
        },

        select: function () {
            this._parent._selectItem(this);
            this._parent._clearTimeouts();
        },

        deselect: function () {
            this._parent._selectItem(null);
        },

        escaped: function () {
            this.select();
        },

        execute: function (options) {
            if (options) {
                Diag.assertParamIsObject(options, "options");
            }

            var owner = this.getOwner();

            // Recovering UI by hiding any popped up sub menu
            // and deselecting the executed item
            this.collapse({ immediate: true });
            owner.escaped();
            this.deselect();
            if (options && options.keepHighlight) {
                this.showHoverHighlight();
            }
            owner._proceedBlur();

            // Checking to see, if the item has an action
            if (this.hasAction()) {
                // Executing the action by merging own arguments and owner arguments. If
                // there is a conflict in arguments, own arguments always win.
                return this.executeAction($.extend({}, owner.getActionArguments(), this.getActionArguments()));
            }
        },

        executeAction: function (args) {

            // Getting action
            var action = this.getAction(), owner = this.getOwner();

            // There might be several types of actions
            if (action) {
                if ($.isFunction(action)) {
                    // If action is a function, executing it by passing
                    // the action arguments as parameters
                    return action.apply(this, [args]);
                }
                else if (action === "navigate" && args.url) {
                    // If the action is "navigate", we check the action arguments
                    // to find a URL. If that satisfies, we are redirecting to that URL.
                    try {
                        if (args.target) {
                            TFSHOST.ActionManager.performAction(TFSHOST.CommonActions.ACTION_WINDOW_OPEN, {
                                url: args.url,
                                target: args.target
                            });
                        }
                        else {
                            TFSHOST.ActionManager.performAction(TFSHOST.CommonActions.ACTION_WINDOW_NAVIGATE, {
                                url: args.url
                            });
                        }
                    }
                    catch (e) {
                        // Eat exception because if the page is modified and the user
                        // presses cancel when prompted about leaving the current page,
                        // an exception is thrown.
                    }
                    return false;
                }
                else if (owner) {
                    return owner.executeAction(new Sys.CommandEventArgs(action, args, this));
                }
                else {

                    // We are executing the command globally which will notify
                    // the subscribers about the action executed.
                    return menuManager.executeCommand(new Sys.CommandEventArgs(action, args, this));
                }
            }
        },

        collapse: function (options) {
            var menu = this._parent,
                immediate = options && options.immediate === true;

            while (menu) {
                // The reason we make this check is menu can be menu item
                // here and it cannot be hidden
                if ($.isFunction(menu.hide)) {
                    menu.hide({ immediate: immediate });
                }

                // Getting the parent
                menu = menu._parent;
            }

            this.getOwner().ownFocus();
        },

        setFocus: function () {
            var element = this._element;
            if (element && this.isEnabled() && !(this.isSeparator() || this.isDefault())) {
                element.addClass("focus");
            }
        },

        removeFocus: function () {
            var element = this._element;
            if (element && this.isEnabled() && !(this.isSeparator() || this.isDefault())) {
                element.removeClass("focus");
            }
        },

        showHoverHighlight: function () {
            /// <summary>Called to show the hover highlight the button</summary>
            this._highlightHover = true;
            this._updateState();
        },

        showPressedHighlight: function () {
            /// <summary>Called to make the button appear to be 'pressed'</summary>
            this._highlightPressed = true;
            this._updateState();
        },

        removePressedHighlight: function () {
            /// <summary>Called to make the button appear to be 'pressed'</summary>
            this._highlightPressed = false;
            this._updateState();
        },

        removeHighlight: function () {
            /// <summary>Called to remove all highlighting on the button</summary>
            this._highlightHover = false;
            this._highlightPressed = false;
            this._updateState();
        },

        _updateState: function () {
            var element = this._element;
            if (element && !(this.isSeparator() || this.isDefault())) {
                if (this.isEnabled()) {
                    element.toggleClass("hover", this._highlightHover);
                    element.toggleClass("pressed", this._highlightPressed);

                    element.removeAttr("disabled");
                    element.removeClass("disabled");
                }
                else {
                    element.attr("disabled", "disabled");
                    element.addClass("disabled");
                    element.removeClass("hover");
                    element.removeClass("pressed");
                }

                element.toggleClass("invisible", this.isHidden());
            }
        },
        updateTitle: function (text) {
            /// <summary>Updates the title of a menu item using either the specified text or
            /// the function provided in the options</summary>
            /// <param name="text" type="String">New title to be displayed</param>
            var newTitle;

            if (typeof (text) === "string") {
                newTitle = text;
            }
            else if (this._item && $.isFunction(this._item.attr('title'))) {
                newTitle = this._item.text.call(this);
            }

            if (newTitle) {
                this._element.attr('title', newTitle);
            }
        },
        updateText: function (text) {
            /// <summary>Updates the text of a menu item using either the specified text or
            /// the function provided in the options</summary>
            /// <param name="text" type="String">New text to be displayed</param>
            var newText;

            if (typeof (text) === "string") {
                newText = text;
            }
            else if (this._item && $.isFunction(this._item.text)) {
                newText = this._item.text.call(this);
            }

            if (newText) {
                // Tries to find the element between the immediate children
                this._element.children("span.text").text(newText);
            }
        },

        getSubMenu: function () {
            var result = null;

            if (this.hasSubMenu()) {
                if (this._children.length === 0) {
                    result = Menu.createIn(this._element, $.extend({}, this._options, { earlyInitialize: false, items: this._item.childItems, type: "submenu", cssCoreClass: "menu sub-menu", tagName: "ul" }));
                    result._parent = this;
                    result._type = MenuType.SubMenu;
                    this._children.push(result);
                }
                else {
                    result = this._children[0];
                }
            }

            return result;
        },

        tryShowSubMenu: function (options) {
            var immediate = options && options.immediate;
            if (!this.isDefault() && this.isEnabled() && this.getSubMenu()) {
                this.showSubMenu({ immediate: immediate });
                return true;
            }
            return false;
        },

        showSubMenu: function (options) {
            var immediate = options && options.immediate === true,
                subMenu = this.getSubMenu();

            assert(Boolean(subMenu), "There is no sub menu to show.");

            subMenu.show({ immediate: immediate, element: this._element, align: this._align });
        },

        hideSubMenu: function (options) {
            if (this.hasSubMenu()) {
                var immediate = options && options.immediate === true,
                   subMenu = this.getSubMenu();

                if (subMenu._parent === this) {
                    subMenu.hide({ immediate: immediate });
                }
            }
        },

        hideSiblings: function (options) {
            this._parent.hideChildren(this, options);
        },

        _onMouseOver: function (e) {
            Diag.logTracePoint("MenuItem._onMouseOver.start");
            var owner = this.getOwner(),
                subMenu, subMenuVisible;

            if (owner._active) {
                subMenuVisible = owner._subMenuVisible;
                this.hideSiblings({ immediate: this._options.immediateShowHide });
                this.select();

                if (owner !== this._parent || subMenuVisible) {
                    owner._subMenuVisible = subMenuVisible;
                    if (this.isEnabled() && this.hasSubMenu()) {
                        subMenu = this.getSubMenu();
                        if (subMenu._parent === this) {
                            if (!subMenu._visible) {
                                this.showSubMenu({ immediate: this._options.immediateShowHide });
                                subMenu.selectDefaultItem();
                            }
                        }
                    }
                }
            }
            else {
                this.showHoverHighlight();
            }

            e.stopPropagation();
        },

        _onMouseOut: function (e) {
            var owner = this.getOwner(),
                subMenu;

            if (owner._active) {
                if (owner !== this._parent) {
                    if (this.hasSubMenu()) {
                        subMenu = this.getSubMenu();
                        if (subMenu._parent === this) {
                            if (!subMenu._visible) {
                                this.deselect();
                                subMenu._clearTimeouts();
                            }
                        }
                        else {
                            this.deselect();
                        }
                    }
                    else {
                        this.deselect();
                    }
                }
            }
            else {
                this.removeHighlight();
            }

            e.stopPropagation();
        },

        _onMouseDown: function (e) {
            this.showPressedHighlight();
        },

        _onMouseUp: function (e) {
            this.removePressedHighlight();
        },

        _onClick: function (e) {

            var subMenu;
            if (this.isEnabled()) {
                if (this.hasSubMenu()) {
                    subMenu = this.getSubMenu();
                    if (subMenu._parent === this && subMenu._visible) {
                        this.getOwner().activate();
                        if (this._options.clickToggles === true) {
                            subMenu.hide({ immediate: true });
                            this.select();
                        }
                        else {
                            subMenu.ownFocus();
                        }
                    }
                    else {
                        if (this.hasAction() && this._options.clickOpensSubMenu === false) {
                            this.select();
                            return this.execute({ keepHighlight: true }) === true;
                        }
                        else {
                            this.getOwner().activate();
                            this.hideSiblings({ immediate: true });
                            this.showSubMenu({ immediate: true });
                            subMenu.selectDefaultItem();
                        }
                    }
                }
                else if (this.hasAction()) {
                    this.select();
                    return this.execute({ keepHighlight: true }) === true;
                }
            }
            else {
                this.getOwner().activate();
                this.hideSiblings({ immediate: true });
                this.select();
            }

            return false;
        },

        _onDropClick: function (e) {
            var subMenu;
            if (this.isEnabled()) {
                this.getOwner().activate();

                if (this.hasSubMenu()) {
                    subMenu = this.getSubMenu();
                    if (subMenu._parent === this && subMenu._visible) {
                        if (this._options.clickToggles === true) {
                            subMenu.hide({ immediate: true });
                            this.select();
                        }
                        else {
                            subMenu.ownFocus();
                        }
                    }
                    else {
                        this.hideSiblings({ immediate: true });
                        this.showSubMenu({ immediate: true });
                        subMenu.selectDefaultItem();
                    }
                }
            }
            return false;
        },

        _onKeyDown: function (e) {
            switch (e.keyCode) {
                case KeyCode.DOWN:
                    this._parent.selectDown();
                    return false;
                case KeyCode.UP:
                    this._parent.selectUp();
                    return false;
                case KeyCode.RIGHT:
                    this._parent.selectRight();
                    return false;
                case KeyCode.LEFT:
                    this._parent.selectLeft();
                    return false;
                case KeyCode.ENTER:
                    if (this.isEnabled()) {
                        if (this.hasAction()) {
                            this.execute();
                        }
                        else if (this.hasSubMenu()) {
                            this.hideSiblings(true);
                            this.showSubMenu(true);
                            this.getSubMenu().selectFirstItem();
                        }
                    }
                    return false;
                case KeyCode.ESCAPE:
                    if (this._parent.escape()) {
                        return false;
                    }

                    break;
                case KeyCode.TAB:
                    if (this.getOwner() === this._parent) {
                        if (e.shiftKey) {
                            if (this._parent.selectPrevItem()) {
                                return false;
                            }
                        }
                        else {
                            if (this._parent.selectNextItem()) {
                                return false;
                            }
                        }
                    }
                    else {
                        this.collapse(true);
                        return this.getOwner()._selectedItem._onKeyDown(e);
                    }
                    break;
            }
        }
    });

    function Menu(options) {
        this.baseConstructor.call(this, $.extend({ cssCoreClass: "menu", tagName: "ul" }, options));

        this._itemsSource = this._options.items;
        this._menuItems = [];
    }

    Menu.extend({
        _typeName: "tfs.menu.menu"
    });

    Menu.inherit(MenuBase, {
        _items: null,           // Child item metadata
        _itemsSource: null,
        _menuItems: null,       // Child menu items of type MenuItem
        _defaultMenuItem: null, // Focus element of this menu
        _selectedItem: null,    // Currently selected item
        _focusItem: null,       // Item which owns the focus
        _childrenCreated: false,
        _popupElement: null,
        _active: false,
        _blockBlur: false,
        _skipUpdateMenuItemStates: false,

        initialize: function () {
            this._base();

            this._decorate();
        },

        _decorate: function () {
            this._element
                .addClass(this._options.cssCoreClass)
                .addClass(this._options.cssClass);

            this._bind("mousedown", delegate(this, this._onMouseDown));
            this._ensureChildren();
        },

        _getItems: function () {
            var that = this, items, finished = false;
            if (!this._items) {
                if (this._itemsSource) {
                    if ($.isFunction(this._itemsSource)) {
                        items = this._itemsSource.call(this, this.getOwner().getContextInfo(), function (items) {
                            if (finished) {
                                if (!that._disposed) {
                                    that.updateItems(items);
                                }
                            }
                            else {
                                finished = true;
                                that._items = items || [];
                            }
                        }, function (error) {
                            var items = [{ text: getErrorMessage(error), icon: "icon-error" }];
                            if (finished) {
                                if (!that._disposed) {
                                    that.updateItems(items);
                                }
                            }
                            else {
                                finished = true;
                                that._items = items;
                            }

                            handleError(error);
                        });

                        if (!finished) {
                            finished = true;
                            if (!items) {
                                items = [{ text: Resources.MenuItemsLoading }];
                            }

                            this._items = items;
                        }
                    }
                    else {
                        this._items = this._itemsSource;
                    }
                }
                else {
                    this._items = [];
                }
            }

            return this._items;
        },

        getItem: function (id) {
            /// <summary>Gets the item which has the specified id</summary>
            /// <param name="tag" type="String">Associated with the menu item</param>
            /// <returns type="MenuItem" />

            var i, len, item;
            for (i = 0, len = this._menuItems.length; i < len; i++) {
                item = this._menuItems[i];
                if (item && id === item.getCommandId()) {
                    return item;
                }
            }

            return null;
        },

        getItemByTag: function (tag) {
            /// <summary>Gets the item which has the specified tag</summary>
            /// <param name="tag" type="String">Associated with the menu item</param>
            /// <returns type="MenuItem" />

            var i, len, item;
            for (i = 0, len = this._menuItems.length; i < len; i++) {
                item = this._menuItems[i];
                if (item && tag === item._item.tag) {
                    return item;
                }
            }
            return null;
        },

        updateCommandStates: function (commands) {
            this._base(commands);

            if (!this._skipUpdateMenuItemStates) {
                this._updateMenuItemStates();
            }
        },

        updateItems: function (items) {
            this._clear();
            this._itemsSource = items;

            if (this._initialized) {
                this._ensureChildren();
                this.updateMenuItemStates();

                // Items are updated, we need to re-position the menu since
                // there might be an overflow
                if ($.isFunction(this._positioningRoutine)) {
                    this._positioningRoutine.call(this);
                }
            }
        },

        _clear: function () {
            this._base();

            this._defaultMenuItem = null;
            this._menuItems = [];
            this._items = null;
            this._childrenCreated = false;
        },

        appendItems: function (appendedItems) {
            if ($.isArray(appendedItems) && appendedItems.length) {
                this.updateItems((this._itemsSource || []).slice(0).concat(appendedItems));
            }
        },

        _enhance: function (element) {
            this._base(element);
            this._enhanceChildren();
        },

        _getMenuItemType: function () {
            return MenuItem;
        },

        _getMenuItemOptions: function (item, extraOptions) {
            return $.extend({
                earlyInitialize: false,
                item: item,
                align: this._getMenuItemAlignment(),
                overflow: this._options.overflow
            }, extraOptions);
        },

        _createChildMenuItem: function (item, menuItemElement) {
            var menuItem, options;

            options = this._getMenuItemOptions(item);

            if (menuItemElement) {
                menuItem = this._getMenuItemType().enhance(menuItemElement, options);
            }
            else {
                menuItem = this._getMenuItemType().createIn(this._element, options);
            }

            // Setting the parent
            menuItem._parent = this;

            // Adding the item to the children
            this._children.push(menuItem);

            if (!menuItem._item.separator) {
                menuItem._index = this._menuItems.length;
                this._menuItems.push(menuItem);
            }

            menuItem.initialize();
        },

        _ensureChildren: function () {
            var that = this;

            if (!this._childrenCreated) {
                $.each(this._getItems(), function (i, item) {
                    that._createChildMenuItem(item);
                });

                this._childrenCreated = true;
            }
        },

        _enhanceChildren: function () {
            var that = this, items;

            items = this._getItems();
            this._element.children("li").each(function (i, li) {
                that._createChildMenuItem($.extend({}, items[i], { decorated: true }), $(li));
            });

            this._childrenCreated = true;
        },

        _getDefaultMenuItem: function () {
            if (!this._defaultMenuItem) {
                var defaultMenuItem = new MenuItem();
                defaultMenuItem._type = MenuType.None;
                defaultMenuItem._parent = this;
                defaultMenuItem._align = MenuAlign.RightBottom;
                defaultMenuItem._index = -1;
                this._children.push(defaultMenuItem);
                this._defaultMenuItem = defaultMenuItem;
            }

            return this._defaultMenuItem;
        },

        _getFirstMenuItem: function () {
            return this._getNextEnabledItem(0);
        },

        _selectItem: function (item, ignoreFocus) {
            if (this._parent) {
                this._parent.select();
            }

            if (this._selectedItem) {
                this._selectedItem.removeHighlight();
                this._selectedItem.removeFocus();
            }

            if (!item) {
                item = this._getDefaultMenuItem();
            }

            item.showHoverHighlight();

            // Setting the item as the selected item
            this._selectedItem = item;
            this._selectedItem.setFocus();
            this.getOwner()._updateAriaAttribute(item);

            if (!ignoreFocus) {
                this.getOwner()._focusItem = item;
            }
        },

        selectDefaultItem: function (ignoreFocus) {
            this._selectItem(null, ignoreFocus);
        },

        selectFirstItem: function () {
            var firstItem = this._getFirstMenuItem();

            if (firstItem) {
                // Selecting the first item
                this._selectItem(firstItem);

                return true;
            }

            return false;
        },

        selectNextItem: function () {
            var index = 0,
                menuItems = this._menuItems,
                selectedItem = this._selectedItem,
                newItem;

            if (menuItems.length > 0) {
                if (selectedItem) {
                    index = selectedItem._index + 1;
                }

                newItem = this._getNextEnabledItem(index);
                if (newItem) {
                    newItem.select();
                    return true;
                }
            }
            return false;
        },

        _getNextEnabledItem: function (index, options) {
            var rollOver = options && options.rollOver === true,
                item,
                menuItems = this._menuItems;

            function getItem(start, end) {
                var i, result;
                for (i = start; i <= end; i++) {
                    result = menuItems[i];
                    if (result.isEnabled()) {
                        return result;
                    }
                }
                return null;
            }

            index = index || 0;

            item = getItem(index, menuItems.length - 1);
            if (item) {
                return item;
            }

            if (rollOver) {
                item = getItem(0, index);
                if (item) {
                    return item;
                }
            }

            return null;
        },

        _getPrevEnabledItem: function (index, options) {
            var rollOver = options && options.rollOver === true,
                item,
                menuItems = this._menuItems;

            function getItem(start, end) {
                var i, result;
                for (i = end; i >= start; i--) {
                    result = menuItems[i];
                    if (result.isEnabled()) {
                        return result;
                    }
                }
                return null;
            }

            index = index || 0;

            item = getItem(0, index);
            if (item) {
                return item;
            }

            if (rollOver) {
                item = getItem(index, menuItems.length - 1);
                if (item) {
                    return item;
                }
            }

            return null;
        },

        selectPrevItem: function () {
            var index = -1,
                menuItems = this._menuItems,
                selectedItem = this._selectedItem,
                newItem;

            if (menuItems.length > 0) {
                if (selectedItem) {
                    index = selectedItem._index - 1;
                }

                newItem = this._getPrevEnabledItem(index);
                if (newItem) {
                    newItem.select();
                    return true;
                }
            }

            return false;
        },

        selectDown: function () {
            var newItem;
            if (this._menuItems.length > 0) {
                newItem = this._getNextEnabledItem(this._selectedItem._index + 1, { rollOver: true });
                if (newItem) {
                    newItem.select();
                    return true;
                }
            }

            return false;
        },

        selectUp: function () {
            var newItem;

            if (this._menuItems.length > 0) {
                newItem = this._getPrevEnabledItem(this._selectedItem._index - 1, { rollOver: true });
                if (newItem) {
                    newItem.select();
                    return true;
                }
            }

            return false;
        },

        selectRight: function (options) {
            if (this._selectedItem.tryShowSubMenu({ immediate: true })) {
                this._selectedItem.getSubMenu().selectFirstItem();
                return true;
            }
            else {
                return this.getOwner().selectRight({ redirected: true });
            }
        },

        selectLeft: function (options) {
            if (this._parent && this._parent._parent !== this.getOwner()) {
                this.hide({ immediate: true });
                this._parent.select();
                return true;
            }
            else {
                return this.getOwner().selectLeft({ redirected: true });
            }
        },

        show: function (options) {
            var immediate = options && options.immediate === true,
                element = options ? options.element : null,
                align = options ? options.align : MenuAlign.RightJustify;

            if (immediate) {
                this._showPopup(element, align);
            }
            else {
                this._startShowTimeout(element, align);
            }
            return true;
        },

        hide: function (options) {
            var immediate = options && options.immediate === true;
            if (immediate) {
                this._hidePopup();
            }
            else {
                this._startHideTimeout();
            }
        },

        hideChildren: function (excludedItem, options) {
            var i, len, item,
                children = this._children,
                immediate = options && options.immediate === true;

            if ($.isArray(children)) {
                for (i = 0, len = children.length; i < len; i++) {
                    item = children[i];
                    if (item !== excludedItem) {
                        item.hideSubMenu({ immediate: immediate });
                    }
                }
            }
        },

        escape: function (options) {
            this.hide({ immediate: true });

            if (this._parent) {
                this._parent.escaped();
            }

            return true;
        },

        ownFocus: function () {
            this._selectItem(this._selectedItem);
        },

        attach: function (parent) {
            if (this._parent !== parent) {
                if (this._visible) {
                    this.hide({ immediate: true });
                }
            }
            this._parent = parent;
        },

        _ensurePopup: function () {
            if (this._ensureInitialized()) {
                this._popupElement = this._element;
            }

            this._ensureChildren();
        },

        _getPopupAlign: function (align) {
            switch (align) {
                case MenuAlign.RightJustify:
                    return ["left-top", "right-top"];
                case MenuAlign.RightBottom:
                    return ["right-top", "right-bottom"];
            }

            return ["left-top", "left-bottom"];
        },

        _getMenuItemAlignment: function () {
            return MenuAlign.RightJustify;
        },

        _showPopup: function (element, align) {
            this._clearTimeouts();
            this._ensurePopup();

            this._pinElement = element;
            this._attachAncestorScroll(element);

            this.updateMenuItemStates();

            var popupAlign = this._getPopupAlign(align);
            this._element.show();

            // We need to have a reference to our positioning routine in case the
            // items are loaded asynchronously. In that case, the popup menu should
            // be repositioned with the new elements (scrollbar might be needed)
            this._positioningRoutine = function () {
                TFS.UI.position(this._element, element, { elementAlign: popupAlign[0], baseAlign: popupAlign[1], overflow: this._options.overflow || "fit-flip", supportScroll: true });
            }

            // Positioning popup menu
            this._positioningRoutine.call(this);

            this._visible = true;
            this.getOwner()._updateSubMenuVisibleState();
            this.ownFocus();
        },

        _hidePopup: function () {
            this._clearTimeouts();
            this._visible = false;

            if (this._pinElement) {
                this._detachAncestorScroll(this._pinElement);
                delete this._pinElement;
            }

            if (this._popupElement) {
                this._popupElement.hide();
            }

            delete this._positioningRoutine;

            this.hideChildren(null, { immediate: true });
            this.getOwner()._updateSubMenuVisibleState();
        },

        _updateMenuItemStates: function () {
            var i, len, child, children = this._children, separator, visible = false;

            for (i = 0, len = children.length; i < len; i++) {
                child = children[i];

                if (child.isSeparator()) {
                    if (separator) {
                        separator.update({ separator: true, hidden: !visible });
                    }

                    separator = child;
                    visible = false;
                } else {
                    if (separator && !child.isHidden()) {
                        visible = true;
                    }

                    child._updateState();
                    child.updateText();
                }
            }

            if (separator) {
                separator.update({ separator: true, hidden: !visible });
            }
        },

        updateMenuItemStates: function () {
            try {
                this._skipUpdateMenuItemStates = true;
                this._fireUpdateCommandStates(this);
            }
            finally {
                this._skipUpdateMenuItemStates = false;
            }

            this._updateMenuItemStates();
        },


        _startShowTimeout: function (element, align) {
            this._clearTimeouts();
            this.delayExecute("show", this.getOwner()._options.showTimeout, true, delegate(this, this._showPopup, [element, align]));
        },

        _startHideTimeout: function () {
            this._clearTimeouts();
            this.delayExecute("hide", this.getOwner()._options.hideTimeout, true, this._hidePopup);
        },

        _clearTimeouts: function () {
            this.cancelDelayedFunction("show");
            this.cancelDelayedFunction("hide");
        },

        executeAction: function (eventArgs) {
            var result;

            if ($.isFunction(this._options.executeAction)) {
                result = this._options.executeAction.call(this, eventArgs);
            }

            if (result !== false) {
                result = menuManager.executeCommand(eventArgs);
            }

            return result;
        },

        _attachAncestorScroll: function (element) {
            this._bind($(element).parents(), "scroll", delegate(this, this._onParentScroll));
        },

        _detachAncestorScroll: function (element) {
            this._unbind($(element).parents(), "scroll");
        },

        _onParentScroll: function (e) {
            this._hidePopup();
        },

        _onMouseDown: function (e) {
            // When the popup menu container is clicked, blur is blocked
            // for a specific period of time (50ms) to be able to catch the click events
            // of the elements inside the popup menu.

            // We need to block blur only for left button click because
            // the execution occurs only for left button click. If we allow blocking blur for
            // right click, it has some side effects regarding to focus (see bug #768719)

            if (e.which === 1) { // 1 => left button
                this._blockBlurUntilTimeout();
            }
        },

        _blockBlurUntilTimeout: function () {
            this._blockBlur = true;
            this.delayExecute("blockBlur", 200, true, function () {
                this._blockBlur = false;
            });
        }
    });

    function MenuOwner(options) {
        this.baseConstructor.call(this, $.extend({ markUnselectable: true, showIcon: true, showTimeout: 500, hideTimeout: 500 }, options));

        this._align = MenuAlign.parse(this._options.align);
    }

    MenuOwner.inherit(Menu, {
        _focusItem: null,       // Focus menu item
        _focusElement: null,    // Element which gets the focus
        _active: false,
        _subMenuVisible: false,

        _canBlur: true,
        _immediateBlur: false,

        setShowIcon: function (showIcon) {
            /// <summary>Sets showIcon option.</summary>
            /// <param name="showIcon" type="Boolean">New state for the showIcon option.</param>
            Diag.assertParamIsBool(showIcon, "showIcon");
            this._options.showIcon = showIcon;
        },

        initialize: function () {
            this._base();
            this._fire("menuInitialized", { menu: this });
        },

        _decorate: function () {
            var focusElem;

            this._base();

            this._focusElement = focusElem = $(domElem("div", "menu-focus-receiver"))
                .attr("tabIndex", 0).attr("role", "menu");

            this._bind(focusElem, "keydown", delegate(this, this._onKeyDown));
            this._bind(focusElem, "focus", delegate(this, this._onFocus));
            this._bind(focusElem, "blur", delegate(this, this._onBlur));
            this._bind(focusElem, "contextmenu", delegate(this, this._onContextMenu));

            this._element.append($(domElem("li", "menu-focus-container")).append(focusElem));

            // TODO: Attach to scroll events to parent elements to hide popup when scroll occurs
            // TODO: Register resize handlers to hide popup when resize occurs
        },

        _updateAriaAttribute: function (item) {
            /// <summary>This is especially necessary for screen readers to read each
            /// row when the selection changes. </summary>
            if (item && item._element) {
                this._focusElement.attr("aria-activedescendant", item._element.attr("id"));
            }
        },

        _getMenuItemAlignment: function () {
            return this._options.popupAlign || this._base();
        },

        _getMenuItemOptions: function (item, extraOptions) {
            return this._base(item, $.extend({ immediateShowHide: true, clickToggles: true, type: "static" }, extraOptions));
        },

        escape: function (options) {
        },

        escaped: function (options) {
        },

        isActive: function () {
            return this._active;
        },

        activate: function () {
            if (!this._activating) {
                this._activating = true;

                try {
                    this._clearBlurTimeout();
                    this._active = true;

                    TFS.UI.tryFocus(this._focusElement);

                    if ($.isFunction(this._options.onActivate)) {
                        this._options.onActivate.call(this);
                    }
                }
                finally {
                    this._activating = false;
                }
            }
        },

        _hide: function () {
            if (this._active) {
                this.hide({ immediate: true });
                this.ownFocus();
            }
        },

        _blur: function () {

            this._clearBlurTimeout();
            this.hide({ immediate: true });

            if (!this._disposed) {
                this._selectItem(null);
            }

            this._active = false;

            if ($.isFunction(this._options.onDeactivate)) {
                this._options.onDeactivate.call(this);
            }
        },

        _updateSubMenuVisibleState: function () {
            var i, len,
                menuItem,
                menuItems = this._menuItems,
                subMenu;

            this._subMenuVisible = false;
            if ($.isArray(menuItems)) {
                for (i = 0, len = menuItems.length; i < len; i++) {
                    menuItem = menuItems[i];
                    if (menuItem.hasSubMenu()) {
                        subMenu = menuItem.getSubMenu();
                        if (subMenu._parent === menuItem && subMenu._visible) {
                            this._subMenuVisible = true;
                            break;
                        }
                    }
                }
            }
        },

        _onKeyDown: function (e) {
            if (this._focusItem) {
                if (this._focusItem._onKeyDown(e) === false) {
                    return false;
                }
            }

            if (e.keyCode === KeyCode.TAB) {
                // We are about to lose focus
                this._immediateBlur = true;
            }
            else if (e.keyCode == 121) { // F10
                if (e.shiftKey) {
                    return this._onContextMenu(e);
                }
            }
        },

        _onFocus: function (e) {
            this.activate();
            if (!this._selectedItem || (this._selectedItem && this._selectedItem._index < 0)) {
                this.selectFirstItem();
            }
        },

        _onBlur: function (e) {
            if (this._immediateBlur) {
                this._immediateBlur = false;
                this._blur();
            }
            else if (this._blockBlur) {
                this._startBlurTimeout();
            }
            else {
                this._blur();
            }
        },

        _proceedBlur: function () {
            this._canBlur = true;
        },

        _startBlurTimeout: function () {
            this._canBlur = false;
            this.delayExecute("blur", TFS.UI.Constants.BlurTimeout, true, function () {
                if (this._canBlur) {
                    this._blur();
                }
                else {
                    this.activate();
                }
            });
        },

        _clearBlurTimeout: function () {
            this.cancelDelayedFunction("blur");
        },

        _onParentScroll: function (e) {
            this._hide();
        },

        _onResize: function (e) {
            this._hide();
        },

        _onContextMenu: function (e) {
            var focusItem = this._focusItem;
            if (focusItem && focusItem.tryShowSubMenu({ immediate: true })) {
                return false;
            }
            return false;
        }
    });

    function MenuBar(options) {
        this.baseConstructor.call(this, $.extend({ cssCoreClass: "menu-bar" }, options));
        this._orientation = MenuOrientation.parse(this._options.orientation);
        this._type = MenuType.Static;
    }

    MenuBar.extend({
        _typeName: "tfs.menu.menubar"
    });

    MenuBar.inherit(MenuOwner, {
        _orientation: MenuOrientation.Horizontal,

        _getMenuItemAlignment: function () {
            return this._orientation === MenuOrientation.Horizontal ? "left-bottom" : "right-justify";
        },

        selectUp: function () {
            return this.base.selectLeft.call(this);
        },

        selectDown: function () {
            return this.base.selectRight.call(this);
        },

        selectLeft: function (options) {
            var subMenuVisible = this._subMenuVisible;

            this.hideChildren(null, { immediate: true });
            this.base.selectUp.call(this);

            if (subMenuVisible) {
                this._subMenuVisible = true;
                if (this._selectedItem.tryShowSubMenu({ immediate: true })) {
                    this._selectedItem.getSubMenu().selectFirstItem();

                    return true;
                }
            }

            return false;
        },

        selectRight: function (options) {
            var subMenuVisible = this._subMenuVisible;

            this.hideChildren(null, { immediate: true });
            this.base.selectDown.call(this);

            if (subMenuVisible) {
                this._subMenuVisible = true;
                if (this._selectedItem.tryShowSubMenu({ immediate: true })) {
                    this._selectedItem.getSubMenu().selectFirstItem();

                    return true;
                }
            }

            return false;
        }
    });

    function PopupMenuItem(options) {
        this.baseConstructor.call(this, $.extend({}, options));
    }

    PopupMenuItem.extend({
        _typeName: "tfs.menu.popup-menuitem"
    });

    PopupMenuItem.inherit(MenuItem, {
        escaped: function () {
            this._base();
            this._parent.escaped();
        }
    });

    function PopupMenu(options) {
        this.baseConstructor.call(this, $.extend({ cssCoreClass: "menu-popup", cssClass: "invisible", hidden: true }, options));
        this._type = MenuType.Popup;
        this._hidden = this._options.hidden === true;
    }

    PopupMenu.extend({
        _typeName: "tfs.menu.popup"
    });

    PopupMenu.inherit(MenuOwner, {
        _align: MenuAlign.RightJustify,
        _floating: false,
        _hidden: false,
        _escapeFocusReceiver: null,

        _getMenuItemType: function () {
            return PopupMenuItem;
        },

        _decorate: function () {
            this._base();
            if (this._hidden === true) {
                this._element.addClass("invisible");
            }
        },

        popup: function (focusElement, pinElement) {
            this._escapeFocusReceiver = focusElement;
            this._floating = true;

            if (this._active) {
                this.escaped();
            }
            else {
                this.activate();

                var firstMenuItem = this._getFirstMenuItem();
                var subMenu = firstMenuItem.getSubMenu();
                if (subMenu) {
                    subMenu.show({ immediate: true, element: pinElement, align: this._align });
                    subMenu.selectDefaultItem();
                }
            }
        },

        selectUp: function () {
            return true;
        },

        selectDown: function (options) {
            var redirected = options && options.redirected;
            if (!redirected) {
                if (this._align !== MenuAlign.RightJustify) {
                    return this.base.selectRight.call(this);
                }
            }

            return false;
        },

        selectLeft: function (options) {
            return true;
        },

        selectRight: function (options) {
            var redirected = options && options.redirected;
            if (!redirected) {
                if (this._align === MenuAlign.RightJustify) {
                    return this.base.selectRight.call(this);
                }
            }

            return false;
        },

        escaped: function () {
            if (this._floating) {
                this._floating = false;

                if (this._escapeFocusReceiver) {
                    this._escapeFocusReceiver.focus();
                    this._escapeFocusReceiver = null;

                    if ($.isFunction(this._options.onPopupEscaped)) {
                        this._options.onPopupEscaped.call(this);
                    }
                }
            }
        }
    });

    function StaticPopupMenu(options) {
        this.baseConstructor.call(this, $.extend({ cssCoreClass: "menu-popup-static", hidden: false }, options));
        this._type = MenuType.DropDown;
        this._hidden = this._options.hidden === true;
    }

    StaticPopupMenu.extend({
        _typeName: "tfs.menu.staticpopup"
    });

    StaticPopupMenu.inherit(PopupMenu, {});

    StaticPopupMenu.registerJQueryWidget("menu", { type: "popup" });


    function SingleCommand(options) {
        /// <summary>Represents an action element which ensures the execution once until command completes
        /// by disabling itself after click (and enables back, when gets enable notification)
        /// <summary>
        this.baseConstructor.call(this, options);
    }

    SingleCommand.inherit(Controls.BaseControl, function () {
        return {
            initialize: function () {
                var options = this._element.data();

                this._base();

                this._bind(options.commandAction || "click", delegate(this, this.onCommandExecute));

                if (options.commandCompleteEvent) {
                    this._bind(window, options.commandCompleteEvent, delegate(this, this.onCommandComplete));
                }
            },

            onCommandExecute: function (e) {
                var options = this._element.data();

                if (options.commandName) {
                    this._element.attr("disabled", true);
                    menuManager.executeCommand(new Sys.CommandEventArgs(options.commandName, options.commandArg, this));
                }
                return false;
            },

            onCommandComplete: function (e) {
                this._element.removeAttr("disabled");
            }
        };

    }());

    MenuBar.registerEnhancement(".menu-bar.enhance");
    PopupMenu.registerEnhancement(".menu-popup.enhance");
    StaticPopupMenu.registerEnhancement(".menu-popup-static.enhance");
    SingleCommand.registerEnhancement(".single-command");

    function sortMenuItems(items) {
        /// <summary>Sort the menu items by rank, pushing those without a rank to the bottom of the list.</summary>

        return items.sort(function (a, b) {
            var leftRank = (a.rank ? a.rank : 9999),
                        rightRank = (b.rank ? b.rank : 9999);

            return leftRank - rightRank;
        });
    }

    return {
        menuManager: menuManager,
        MenuBar: MenuBar,
        PopupMenu: PopupMenu,
        StaticPopupMenu: StaticPopupMenu,
        MenuItemState: MenuItemState,
        sortMenuItems: sortMenuItems
    };
});

// SIG // Begin signature block
// SIG // MIIaswYJKoZIhvcNAQcCoIIapDCCGqACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFE6FHQSLVCmP
// SIG // CgktSA4oSPTfkJsfoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSmMIIEogIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIHIMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQHYtGcZLH400pzKFkM2z8c
// SIG // 2bBTozBoBgorBgEEAYI3AgEMMVowWKA+gDwAVABGAFMA
// SIG // LgBVAEkALgBDAG8AbgB0AHIAbwBsAHMALgBNAGUAbgB1
// SIG // AHMALgBkAGUAYgB1AGcALgBqAHOhFoAUaHR0cDovL21p
// SIG // Y3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEAIAcR
// SIG // XHgZDOhIEDSky8eRR7KB8Y3u0mCNHCzZ6Eg46nnLu7HJ
// SIG // MX4dMNIyb0Lr2CjIheVp6nhfzo3WPhxPDPfuAj50hOVM
// SIG // sWaO+jrEcSM+tJbn38DzW7aturFlmV83rPqMUlxxc3N7
// SIG // tgtCbDXNgbWRTteHu/XI6XcpQxDff3jB/VkSZmplPtO5
// SIG // REb9deMf2FELD5VtxGpUPRnImjjbZ2NOyqsfT/3i6Il3
// SIG // ggRmDudnhaC6ggpQtek7MFkZlPDC7A7P/INtsJQTbx4H
// SIG // 4+h2de6grw9VxZIEgIg2D4FibgStDCGm1Uz49P04CqLU
// SIG // DTu+lCxzdI0ciWEr7zXYczaiRtW1uKGCAh8wggIbBgkq
// SIG // hkiG9w0BCQYxggIMMIICCAIBATCBhTB3MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0ECCmECjkIAAAAAAB8wCQYFKw4DAhoF
// SIG // AKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJ
// SIG // KoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZI
// SIG // hvcNAQkEMRYEFHEMp++fdhapEueOyG95NxF5rMh1MA0G
// SIG // CSqGSIb3DQEBBQUABIIBAC7Mff5A/AP3FSoCmPqN6V3M
// SIG // LaiW+DJkix6SwbTvbsvZG7IKh+4QEpqo7q8lWks73r9m
// SIG // j/06Y6xlCB6tpVoo+JXM2Wm/8GY4qZi8mJ4l5N89lxo5
// SIG // rIPn6m/DBIru/x2LyFGjbIblJyjZZD7CmqXU7vME+8Mu
// SIG // YS9oMBi5Tyug5aYfiQXRb3FbzEEUDWM8cE5L1meOdeX3
// SIG // 9qDKTRo+gXIY5oeKdSLB9F0AYAQ8KyZh8YRfzpxSrCbg
// SIG // b5t8+R846d+7EAP1WCHNGvAp84sqdVmBeTZthNN0TkbK
// SIG // KKJAt9D6K/rITZIWL3FPcW6/+qV+8T9lKN+C1IPU7s5b
// SIG // DuhSgOIR43A=
// SIG // End signature block
